'use strict';

/* --------------------------------------------------------------
 orders_overview_columns.js 2020-01-13
 Gambio GmbH
 http://www.gambio.de
 Copyright (c) 2020 Gambio GmbH
 Released under the GNU General Public License (Version 2)
 [http://www.gnu.org/licenses/gpl-2.0.html]
 --------------------------------------------------------------
 */

jse.libs.orders_overview_columns = jse.libs.orders_overview_columns || {};

/**
 * ## Orders Table Column Definitions
 *
 * This module defines the column definition of the order overview table. They can be overridden by other
 * scripts by modifying the array with new columns, or by replacing the property values of the contained
 * fields.
 *
 * @module Admin/Libs/orders_overview_columns
 * @exports jse.libs.orders_overview_columns
 * @requires momentjs
 */
(function (exports) {

    'use strict';

    exports.checkbox = exports.checkbox || {
        data: null,
        minWidth: '50px',
        widthFactor: 0.01,
        orderable: false,
        searchable: false,
        defaultContent: '<input type="checkbox" />'
    };

    exports.number = exports.number || {
        data: 'number',
        minWidth: '75px',
        widthFactor: 1,
        className: 'numeric',
        render: function render(data, type, full, meta) {
            var linkElement = '';

            if (full.DT_RowData.comment !== '') {
                // Remove double quotes to avoid broken tooltips
                var escapedComment = full.DT_RowData.comment.replace(/"/, '');

                linkElement += '\n\t\t\t\t\t\t\t\t\t<i class="fa fa-comment-o tooltip-order-comment tooltip-trigger"\n\t\t\t\t\t\t\t\t\t\taria-hidden="true" title="' + escapedComment + '"></i>&nbsp;\n\t\t\t\t\t\t\t\t';
            }

            var editUrl = 'orders.php?' + $.param({
                oID: full.DT_RowData.id,
                action: 'edit',
                overview: $.deparam(window.location.search.slice(1))
            });

            linkElement += '\n\t\t\t\t\t\t\t\t<a class="tooltip-order-items" href="' + editUrl + '">\n\t\t\t\t\t\t\t\t\t' + full.DT_RowData.id + '\n\t\t\t\t\t\t\t\t</a>\n\t\t\t\t\t\t\t';

            return linkElement;
        }
    };

    exports.customer = exports.customer || {
        data: 'customer',
        minWidth: '190px',
        widthFactor: 1.5,
        render: function render(data, type, full, meta) {
            var linkElement = full.DT_RowData.customerId ? '<a class="tooltip-customer-addresses" \n\t\t\t\t\t\t\thref="customers.php?cID=' + full.DT_RowData.customerId + '&action=edit">' + data + '</a>' : '<span class="tooltip-customer-addresses">' + data + '</span>';

            if (full.DT_RowData.customerMemos.length > 0) {
                linkElement += ' <i class="fa fa-sticky-note-o tooltip-customer-memos tooltip-trigger" \n                                aria-hidden="true"></i>';
            }

            return linkElement;
        }
    };

    exports.group = exports.group || {
        data: 'group',
        minWidth: '85px',
        widthFactor: 1.2
    };

    exports.sum = exports.sum || {
        data: 'sum',
        minWidth: '90px',
        widthFactor: 1,
        className: 'numeric',
        render: function render(data, type, full, meta) {
            return '<span class="tooltip-order-sum-block">' + data + '</span>';
        }
    };

    exports.paymentMethod = exports.paymentMethod || {
        data: 'paymentMethod',
        minWidth: '110px',
        widthFactor: 2,
        render: function render(data, type, full, meta) {
            return '<span title="' + full.DT_RowData.paymentMethod + '">' + data + '</span>';
        }
    };

    exports.shippingMethod = exports.shippingMethod || {
        data: 'shippingMethod',
        minWidth: '110px',
        widthFactor: 2,
        className: 'shipping-method',
        render: function render(data, type, full, meta) {
            return '<span title="' + full.DT_RowData.shippingMethod + '">' + data + '</span>' + (full.DT_RowData.trackingLinks.length ? ' <i class="fa fa-truck fa-lg tooltip-tracking-links tooltip-trigger"></i>' : '');
        },
        createdCell: function createdCell(td, cellData, rowData, row, col) {
            rowData.DT_RowData.trackingLinks.length ? $(td).children(':first').data('orderId', rowData.DT_RowData.id).attr('data-toggle', 'modal').attr('data-target', '.add-tracking-number.modal') : $(td).data('orderId', rowData.DT_RowData.id).attr('data-toggle', 'modal').attr('data-target', '.add-tracking-number.modal');
        }
    };

    exports.countryIsoCode = exports.countryIsoCode || {
        data: 'countryIsoCode',
        minWidth: '75px',
        widthFactor: 1.4,
        render: function render(data, type, full, meta) {
            var html = '';

            if (data) {
                html = '<span class="flag-icon flag-icon-' + data.toLowerCase() + '"></span>&nbsp;';
            }

            var title = jse.core.lang.translate('SHIPPING_ORIGIN_COUNTRY_TITLE', 'configuration') + ': ' + full.DT_RowData.country;

            html += '<span title="' + title + '">' + data + '</span>';

            return html;
        }
    };

    exports.date = exports.date || {
        data: 'date',
        minWidth: '100px',
        widthFactor: 1.6,
        render: function render(data, type, full, meta) {
            return moment(data).format('DD.MM.YY - HH:mm');
        }
    };

    exports.status = exports.status || {
        data: 'status',
        minWidth: '120px',
        widthFactor: 2,
        render: function render(data, type, full, meta) {
            return '\n\t\t\t\t\t<span data-toggle="modal" data-target=".status.modal"\n\t\t\t\t\t\t\tclass="order-status tooltip-order-status-history label label-' + full.DT_RowData.statusId + '">\n\t\t\t\t\t\t' + data + '\n\t\t\t\t\t</span>\n\t\t\t\t';
        }
    };

    exports.totalWeight = exports.totalWeight || {
        data: 'totalWeight',
        minWidth: '50px',
        widthFactor: 0.6,
        className: 'numeric'
    };

    exports.invoiceNumber = exports.invoiceNumber || {
        data: 'invoiceNumber',
        minWidth: '75px',
        widthFactor: 1,
        render: function render(data, type, full, meta) {
            // Create a 'span' container element.
            var $html = document.createElement('span');

            // Iterator function to add a link element into container.
            var _addLinkElement = function _addLinkElement(invoiceNumber, index, array) {
                // Is the current iteration the last one?
                var isLastIteration = index === array.length - 1;

                // Invoice link parameters.
                var parameters = {
                    module: 'OrderAdmin',
                    action: 'showPdf',
                    type: 'invoice',
                    order_id: full.DT_RowData.id,
                    invoice_number: invoiceNumber
                };

                // Compound invoice link.
                var url = jse.core.config.get('appUrl') + '/admin/request_port.php?' + $.param(parameters);

                // Create link element.
                var $link = document.createElement('a');

                // Set link on element.
                $link.setAttribute('href', url);
                $link.setAttribute('target', '_blank');

                // Set invoice number as text on element.
                $link.textContent = invoiceNumber + (isLastIteration ? '' : ', ');

                // Append element to container.
                $html.appendChild($link);
            };

            // Add tooltip classes to element.
            $html.classList.add('tooltip-invoices', 'tooltip-trigger');

            // Iterate over each invoice number and create link.
            full.DT_RowData.invoiceNumbers.forEach(_addLinkElement);

            return $html.outerHTML;
        }
    };

    exports.actions = exports.actions || {
        data: null,
        minWidth: '350px',
        widthFactor: 4.6,
        className: 'actions',
        orderable: false,
        searchable: false,
        render: function render(data, type, full, meta) {
            var withdrawalIdsHtml = '';

            full.DT_RowData.withdrawalIds.forEach(function (withdrawalId) {
                withdrawalIdsHtml += '\n\t\t\t\t\t\t<a href="admin.php?do=Withdrawals&id=' + withdrawalId + '"\n\t\t\t\t\t\t\t\ttitle="' + jse.core.lang.translate('TABLE_HEADING_WITHDRAWAL_ID', 'orders') + ' ' + withdrawalId + '">\n\t\t\t\t\t\t\t<img src="html/assets/images/legacy/icons/withdrawal-on.png" \n\t\t\t\t\t\t\t\tclass="tooltip-withdrawal tooltip-trigger meta-icon" \n\t\t\t\t\t\t\t\tdata-withdrawal-id="' + withdrawalId + '" />\n\t\t\t\t\t\t</a>\n\t\t\t\t\t';
            });

            var editUrl = 'orders.php?' + $.param({
                oID: full.DT_RowData.id,
                action: 'edit',
                overview: $.deparam(window.location.search.slice(1))
            });

            var mailStatusHtml = !full.DT_RowData.mailStatus ? '<i class="fa fa-envelope-o meta-icon tooltip-confirmation-not-sent email-order tooltip-trigger"\n\t\t\t\t\t\ttitle="' + jse.core.lang.translate('TEXT_CONFIRMATION_NOT_SENT', 'orders') + '"></i>' : '';

            return '\n\t\t\t\t\t<div class="pull-left">\n\t\t\t\t\t\t' + withdrawalIdsHtml + '\n\t\t\t\t\t\t' + mailStatusHtml + '\n\t\t\t\t\t</div>\n\t\t\t\t\t\n\t\t\t\t\t<div class="pull-right action-list visible-on-hover">\n\t\t\t\t\t\t<a href="' + editUrl + '">\n\t\t\t\t\t\t\t<i class="fa fa-eye edit"></i>\n\t\t\t\t\t\t</a>\n\t\t\t\t\t\t\n\t\t\t\t\t\t<div class="btn-group dropdown">\n\t\t\t\t\t\t\t<button type="button"\n\t\t\t\t\t\t\t\t\tclass="btn btn-default"></button>\n\t\t\t\t\t\t\t<button type="button"\n\t\t\t\t\t\t\t\t\tclass="btn btn-default dropdown-toggle"\n\t\t\t\t\t\t\t\t\tdata-toggle="dropdown"\n\t\t\t\t\t\t\t\t\taria-haspopup="true"\n\t\t\t\t\t\t\t\t\taria-expanded="false">\n\t\t\t\t\t\t\t\t<span class="caret"></span>\n\t\t\t\t\t\t\t\t<span class="sr-only">Toggle Dropdown</span>\n\t\t\t\t\t\t\t</button>\n\t\t\t\t\t\t\t<ul class="dropdown-menu dropdown-menu-right"></ul>\n\t\t\t\t\t\t</div>\n\t\t\t\t\t</div>\n\t\t\t\t';
        }
    };
})(jse.libs.orders_overview_columns);
//# sourceMappingURL=data:application/json;charset=utf8;base64,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
